#version 430 core
layout (location = 0) in vec3 fragPosition;
layout (location = 1) in vec3 noiseCoords;
layout (location = 2) in vec3 fragNormal;
layout (location = 3) in vec2 staticNoiseCoords;

layout (location = 0) out vec4 gAlbedoAlpha;

uniform float time;

float hash(float n) { return fract(sin(n) * 1e4); }
float hash(vec2 p) { return fract(1e4 * sin(17.0 * p.x + p.y * 0.1) * (0.1 + abs(sin(p.y * 13.0 + p.x)))); }

float noise(vec3 x) {
	const vec3 step = vec3(110, 241, 171);

	vec3 i = floor(x);
	vec3 f = fract(x);
 
	// For performance, compute the base input to a 1D hash from the integer part of the argument and the 
	// incremental change to the 1D based on the 3D -> 1D wrapping
    float n = dot(i, step);

	vec3 u = f * f * (3.0 - 2.0 * f);
	return mix(mix(mix( hash(n + dot(step, vec3(0, 0, 0))), hash(n + dot(step, vec3(1, 0, 0))), u.x),
                   mix( hash(n + dot(step, vec3(0, 1, 0))), hash(n + dot(step, vec3(1, 1, 0))), u.x), u.y),
               mix(mix( hash(n + dot(step, vec3(0, 0, 1))), hash(n + dot(step, vec3(1, 0, 1))), u.x),
                   mix( hash(n + dot(step, vec3(0, 1, 1))), hash(n + dot(step, vec3(1, 1, 1))), u.x), u.y), u.z);
}

#define NUM_OCTAVES 4

float fbm ( in vec3 _st) {
    float v = 0.0;
    float a = 0.5;
    vec3 shift = vec3(100.0);
    // Rotate to reduce axial bias
    mat2 rot = mat2(cos(0.5), sin(0.5),
                    -sin(0.5), cos(0.50));
    for (int i = 0; i < NUM_OCTAVES; ++i) {
        v += a * noise(_st);
        _st = _st * 2.0 + shift;
        a *= 0.5;
    }
    return v;
}

float linearize_depth(float d,float zNear,float zFar)
{
  float z_n = 2.0 * d - 1.0;
  return 2.0 * zNear * zFar / (zFar + zNear - z_n * (zFar - zNear));
}

uniform int iterationCount;
uniform float windVelocityMagnitude;
uniform float cutoffAmount;
uniform vec3 viewDir;
uniform mat4 VP;
uniform mat4 VPW;

uniform mat4 worldMatrix;

uniform vec3 cameraPosition;

uniform vec3 cloudColorPrimary = vec3(1.0, 0.0, 0.0);
uniform vec3 cloudColorSecondary = vec3(0.0, 0.0, 1.0);

uniform int screenWidth;
uniform int screenHeight;

uniform int stepCount;

struct Light {
  vec4 PositionLinear;
  vec4 ColorQuadratic;
	vec4 IntensityPADDING_PADDING_PADDING;
};

layout (std140, binding = 0) uniform LightBlock {
  Light lights[1];
};

layout (binding = 0) uniform sampler2D depthTexture; 

uniform sampler2D noiseTex;

vec4 integrate( in vec4 sum, in float den, float atmoDepth, vec3 rayPos, vec3 normal, float stepDistance)
{
	vec3 albedo = mix( cloudColorPrimary, cloudColorSecondary, atmoDepth );

	vec3 lighting = vec3(0.0);

	vec3 N = normal;

	vec3 L = normalize(lights[0].PositionLinear.xyz - rayPos);
	float distance = length(lights[0].PositionLinear.xyz - rayPos);
	float attenuation = (1.0) / (1.0 + lights[0].PositionLinear.w * distance + lights[0].ColorQuadratic.w * distance * distance);
	vec3 radiance = lights[0].ColorQuadratic.xyz * attenuation;

	float NdotL = max(dot(N, L), 0.0);    

	lighting = radiance * lights[0].IntensityPADDING_PADDING_PADDING.x * NdotL;

	vec3 ambient = vec3(0.1);

	vec4 col = vec4((lighting + ambient) * albedo, den);

	float stepDistanceTransparency = 0.4 * (stepDistance / 2.0);

    // front to back blending    
    col.a *= stepDistanceTransparency;
    col.rgb *= col.a;
    return sum + col*(1.0-sum.a);
}


void main() {

	vec3 derp = vec3(1.0);

	float transparency = 0.0;
	
	vec3 rayPosition = fragPosition;
	float tempTransparency = 0.0;

	vec3 rayDirection = normalize(fragPosition - cameraPosition);

	vec4 sum = vec4(0.0);

	float baseStepDistance = 95.92 / float(stepCount);
	float stepDistance = baseStepDistance;

	for(int i = 0; i < stepCount; i++){
	
		rayPosition += rayDirection * stepDistance;

		vec4 rayInVP = VPW * vec4(rayPosition, 1.0);

		rayInVP.xyz = (rayInVP.xyz / rayInVP.w) * 0.5 + 0.5;

		float normalizedRayDepth = linearize_depth(rayInVP.z, 1.0, 6000);  

		gAlbedoAlpha = vec4(derp, 1.0);
		 
		float cutoffAddition = 1.0;
		float rayPosLength = length(rayPosition - worldMatrix[3].xyz);
		
		if(rayPosLength > 120.0){
		
			cutoffAddition = 1.0 - (rayPosLength - 120) / 10.0;
		
		}
		
		if(rayPosLength < 110.0){
		
			cutoffAddition = 1.0 - (110 - rayPosLength) / 10.0;
			
			cutoffAddition = clamp(cutoffAddition, 0.0, 1.0);
		}
		
		if(cutoffAddition == 0.0 || texture(depthTexture, rayInVP.xy).w < normalizedRayDepth){
			break;
		}else{

			tempTransparency = fbm((rayPosition) / 30.0f);
			tempTransparency -= cutoffAmount;
			tempTransparency *= 1 / (1 - cutoffAmount);
			tempTransparency = clamp(tempTransparency, 0.0, 1.0);

			float normalizedAtmoDepth = (rayPosLength - 100) / 30.0f;

			vec3 normal = normalize(rayPosition - worldMatrix[3].xyz);

			sum = integrate(sum, tempTransparency * cutoffAddition, normalizedAtmoDepth, rayPosition, normal, stepDistance); 

			stepDistance = baseStepDistance;

		}

	}

	gAlbedoAlpha = sum;

}
